/* This is an implementation of the Unlambda programming language. */
/* This one is in C (the Master One is in Scheme). */

/* Copyright (C) 1999 by David A. Madore <david.madore@ens.fr> */

/* This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* This is a more or less mechanical translation of the Java version
 * of the interpreter.  Please refer to the latter for all
 * explanations about how things work. */

/* This is made to work with Hans Boehm's famous conservative C/C++
 * garbage collector.  If you don't want to use this, replace <gc.h>
 * by <stdlib.h> below, and replace all calls to GC_malloc() by
 * malloc().  Of course, in that way, memory used will never be
 * reclaimed and you will run out quickly - however, the interpreter
 * should still function correctly. */

#include <stdio.h>
#include <gc.h>

struct function_s {
  enum {
    FUNCTION_I,
    FUNCTION_DOT,
    FUNCTION_K1,
    FUNCTION_K,
    FUNCTION_S2,
    FUNCTION_S1,
    FUNCTION_S,
    FUNCTION_V,
    FUNCTION_D1,
    FUNCTION_D,
    FUNCTION_CONT,
    FUNCTION_C,
    FUNCTION_E,
    FUNCTION_AT,
    FUNCTION_QUES,
    FUNCTION_PIPE
  } t;
  union {
    char function_dot_v;
    struct function_s *function_k1_v;
    struct {
      struct function_s *x, *y;
    } function_s2_v;
    struct function_s *function_s1_v;
    struct expression_s *function_d1_v;
    struct continuation_s *function_cont_v;
    char function_ques_v;
  } d;
};

struct expression_s {
  enum {
    EXPRESSION_FUNCTION,
    EXPRESSION_APPLICATION
  } t;
  union {
    struct function_s *expression_function_v;
    struct {
      struct expression_s *rator, *rand;
    } expression_application_v;
  } d;
};

struct continuation_s {
  enum {
    CONTINUATION_APP1,
    CONTINUATION_APP,
    CONTINUATION_DEL,
    CONTINUATION_FINAL
  } t;
  union {
    struct {
      struct expression_s *rand;
      struct continuation_s *cont;
    } continuation_app1_v;
    struct {
      struct function_s *erator;
      struct continuation_s *cont;
    } continuation_app_v;
    struct {
      struct function_s *erand;
      struct continuation_s *cont;
    } continuation_del_v;
  } d;
};

struct task_s {
  enum {
    TASK_EVAL,
    TASK_APP1,
    TASK_APP,
    TASK_FINAL
  } t;
  union {
    struct {
      struct expression_s *expr;
      struct continuation_s *cont;
    } task_eval_v;
    struct {
      struct function_s *erator;
      struct expression_s *rand;
      struct continuation_s *cont;
    } task_app1_v;
    struct {
      struct function_s *erator, *erand;
      struct continuation_s *cont;
    } task_app_v;
  } d;
};

char current_ch = EOF;

struct task_s *
invoke (struct continuation_s *cont, struct function_s *val)
{
  switch ( cont->t )
    {
    case CONTINUATION_APP1:
      {
	struct task_s *task = GC_malloc (sizeof (struct task_s));

	task->t = TASK_APP1;
	task->d.task_app1_v.erator = val;
	task->d.task_app1_v.rand = cont->d.continuation_app1_v.rand;
	task->d.task_app1_v.cont = cont->d.continuation_app1_v.cont;
	return task;
      }
    case CONTINUATION_APP:
      {
	struct task_s *task = GC_malloc (sizeof (struct task_s));

	task->t = TASK_APP;
	task->d.task_app_v.erator = cont->d.continuation_app_v.erator;
	task->d.task_app_v.erand = val;
	task->d.task_app_v.cont = cont->d.continuation_app_v.cont;
	return task;
      }
    case CONTINUATION_DEL:
      {
	struct task_s *task = GC_malloc (sizeof (struct task_s));

	task->t = TASK_APP;
	task->d.task_app_v.erator = val;
	task->d.task_app_v.erand = cont->d.continuation_del_v.erand;
	task->d.task_app_v.cont = cont->d.continuation_del_v.cont;
	return task;
      }
    case CONTINUATION_FINAL:
      {
	struct task_s *task = GC_malloc (sizeof (struct task_s));

	task->t = TASK_FINAL;
	return task;
      }
    }
  fprintf (stderr, "INTERNAL ERROR: invoke() surprised!\n");
  return NULL;
}

struct task_s *
apply (struct function_s *rator, struct function_s *rand,
       struct continuation_s *cont)
{
  switch ( rator->t )
    {
    case FUNCTION_I:
      return invoke (cont, rand);
    case FUNCTION_DOT:
      putchar (rator->d.function_dot_v);
      return invoke (cont, rand);
    case FUNCTION_K1:
      return invoke (cont, rator->d.function_k1_v);
    case FUNCTION_K:
      {
	struct function_s *val = GC_malloc (sizeof (struct function_s));

	val->t = FUNCTION_K1;
	val->d.function_k1_v = rand;
	return invoke (cont, val);
      }
    case FUNCTION_S2:
      {
	struct expression_s *e_x = GC_malloc (sizeof (struct expression_s));
	struct expression_s *e_y = GC_malloc (sizeof (struct expression_s));
	struct expression_s *e_z = GC_malloc (sizeof (struct expression_s));
	struct expression_s *e1 = GC_malloc (sizeof (struct expression_s));
	struct expression_s *e2 = GC_malloc (sizeof (struct expression_s));
	struct expression_s *e = GC_malloc (sizeof (struct expression_s));
	struct task_s *task = GC_malloc (sizeof (struct task_s));

	e_x->t = EXPRESSION_FUNCTION;
	e_x->d.expression_function_v = rator->d.function_s2_v.x;
	e_y->t = EXPRESSION_FUNCTION;
	e_y->d.expression_function_v = rator->d.function_s2_v.y;
	e_z->t = EXPRESSION_FUNCTION;
	e_z->d.expression_function_v = rand;
	e1->t = EXPRESSION_APPLICATION;
	e1->d.expression_application_v.rator = e_x;
	e1->d.expression_application_v.rand = e_z;
	e2->t = EXPRESSION_APPLICATION;
	e2->d.expression_application_v.rator = e_y;
	e2->d.expression_application_v.rand = e_z;
	e->t = EXPRESSION_APPLICATION;
	e->d.expression_application_v.rator = e1;
	e->d.expression_application_v.rand = e2;
	task->t = TASK_EVAL;
	task->d.task_eval_v.expr = e;
	task->d.task_eval_v.cont = cont;
	return task;
      }
    case FUNCTION_S1:
      {
	struct function_s *val = GC_malloc (sizeof (struct function_s));

	val->t = FUNCTION_S2;
	val->d.function_s2_v.x = rator->d.function_s1_v;
	val->d.function_s2_v.y = rand;
	return invoke (cont, val);
      }
    case FUNCTION_S:
      {
	struct function_s *val = GC_malloc (sizeof (struct function_s));

	val->t = FUNCTION_S1;
	val->d.function_s1_v = rand;
	return invoke (cont, val);
      }
    case FUNCTION_V:
      return invoke (cont, rator);
    case FUNCTION_D1:
      {
	struct continuation_s *ncont = GC_malloc (sizeof (struct continuation_s));
	struct task_s *task = GC_malloc (sizeof (struct task_s));

	ncont->t = CONTINUATION_DEL;
	ncont->d.continuation_del_v.erand = rand;
	ncont->d.continuation_del_v.cont = cont;
	task->t = TASK_EVAL;
	task->d.task_eval_v.expr = rator->d.function_d1_v;
	task->d.task_eval_v.cont = ncont;
	return task;
      }
    case FUNCTION_D:
      {
	struct expression_s *promise = GC_malloc (sizeof (struct expression_s));
	struct function_s *val = GC_malloc (sizeof (struct function_s));

	promise->t = EXPRESSION_FUNCTION;
	promise->d.expression_function_v = rand;
	val->t = FUNCTION_D1;
	val->d.function_d1_v = promise;
	return invoke (cont, val);
      }
    case FUNCTION_CONT:
      return invoke (rator->d.function_cont_v, rand);
    case FUNCTION_C:
      {
	struct function_s *val = GC_malloc (sizeof (struct function_s));
	struct task_s *task = GC_malloc (sizeof (struct task_s));

	val->t = FUNCTION_CONT;
	val->d.function_cont_v = cont;
	task->t = TASK_APP;
	task->d.task_app_v.erator = rand;
	task->d.task_app_v.erand = val;
	task->d.task_app_v.cont = cont;
	return task;
      }
    case FUNCTION_E:
      {
	struct task_s *task = GC_malloc (sizeof (struct task_s));

	task->t = TASK_FINAL;
	return task;
      }
    case FUNCTION_AT:
      {
	struct function_s *val = GC_malloc (sizeof (struct function_s));
	struct task_s *task = GC_malloc (sizeof (struct task_s));

	current_ch = getchar ();
	val->t = (current_ch != EOF ? FUNCTION_I : FUNCTION_V);
	task->t = TASK_APP;
	task->d.task_app_v.erator = rand;
	task->d.task_app_v.erand = val;
	task->d.task_app_v.cont = cont;
	return task;
      }
    case FUNCTION_QUES:
      {
	struct function_s *val = GC_malloc (sizeof (struct function_s));
	struct task_s *task = GC_malloc (sizeof (struct task_s));

	val->t = (current_ch == rator->d.function_ques_v
		  ? FUNCTION_I : FUNCTION_V);
	task->t = TASK_APP;
	task->d.task_app_v.erator = rand;
	task->d.task_app_v.erand = val;
	task->d.task_app_v.cont = cont;
	return task;
      }
    case FUNCTION_PIPE:
      {
	struct function_s *val = GC_malloc (sizeof (struct function_s));
	struct task_s *task = GC_malloc (sizeof (struct task_s));

	if ( current_ch != EOF )
	  {
	    val->t = FUNCTION_DOT;
	    val->d.function_dot_v = current_ch;
	  }
	else
	  val->t = FUNCTION_V;
	task->t = TASK_APP;
	task->d.task_app_v.erator = rand;
	task->d.task_app_v.erand = val;
	task->d.task_app_v.cont = cont;
	return task;
      }
    }
  fprintf (stderr, "INTERNAL ERROR: apply() surprised!\n");
  return NULL;
}

struct task_s *
eval (struct expression_s *expr, struct continuation_s *cont)
{
  switch ( expr->t )
    {
    case EXPRESSION_FUNCTION:
      return invoke (cont, expr->d.expression_function_v);
    case EXPRESSION_APPLICATION:
      {
	struct continuation_s *ncont = GC_malloc (sizeof (struct continuation_s));
	struct task_s *task = GC_malloc (sizeof (struct task_s));

	ncont->t = CONTINUATION_APP1;
	ncont->d.continuation_app1_v.rand
	  = expr->d.expression_application_v.rand;
	ncont->d.continuation_app1_v.cont = cont;
	task->t = TASK_EVAL;
	task->d.task_eval_v.expr
	  = expr->d.expression_application_v.rator;
	task->d.task_eval_v.cont = ncont;
	return task;
      }
    }
  fprintf (stderr, "INTERNAL ERROR: eval() surprised!\n");
  return NULL;
}

struct task_s *
run (struct task_s *task)
{
  switch ( task->t )
    {
    case TASK_EVAL:
      return eval (task->d.task_eval_v.expr, task->d.task_eval_v.cont);
    case TASK_APP1:
      {
	if ( task->d.task_app1_v.erator->t == FUNCTION_D )
	  {
	    struct function_s *val = GC_malloc (sizeof (struct function_s));

	    val->t = FUNCTION_D1;
	    val->d.function_d1_v = task->d.task_app1_v.rand;
	    return invoke (task->d.task_app1_v.cont, val);
	  }
	else
	  {
	    struct continuation_s *ncont = GC_malloc (sizeof (struct continuation_s));

	    ncont->t = CONTINUATION_APP;
	    ncont->d.continuation_app_v.erator = task->d.task_app1_v.erator;
	    ncont->d.continuation_app_v.cont = task->d.task_app1_v.cont;
	    return eval (task->d.task_app1_v.rand, ncont);
	  }
      }
    case TASK_APP:
      return apply (task->d.task_app_v.erator, task->d.task_app_v.erand,
		    task->d.task_app_v.cont);
    case TASK_FINAL:
      /* Should not happen */;
    }
  fprintf (stderr, "INTERNAL ERROR: run() surprised!\n");
  return NULL;
}

struct expression_s *
parse (FILE *input)
{
  int ch;
  do {
    ch = getc (input);
    if ( ch == '#' )
      while ( ch != '\n' && ch != EOF )
	ch = getc (input);
  } while ( ch == ' ' || ch == '\n' || ch == '\r' || ch == '\t' );
  if ( ch == '`' )
    {
      struct expression_s *rator = parse (input);
      struct expression_s *rand = parse (input);
      struct expression_s *expr = GC_malloc (sizeof (struct expression_s));

      expr->t = EXPRESSION_APPLICATION;
      expr->d.expression_application_v.rator = rator;
      expr->d.expression_application_v.rand = rand;
      return expr;
    }
  else if ( ch == 'i' || ch == 'I' )
    {
      struct function_s *fun = GC_malloc (sizeof (struct function_s));
      struct expression_s *expr = GC_malloc (sizeof (struct expression_s));

      fun->t = FUNCTION_I;
      expr->t = EXPRESSION_FUNCTION;
      expr->d.expression_function_v = fun;
      return expr;
    }
  else if ( ch == 'k' || ch == 'K' )
    {
      struct function_s *fun = GC_malloc (sizeof (struct function_s));
      struct expression_s *expr = GC_malloc (sizeof (struct expression_s));

      fun->t = FUNCTION_K;
      expr->t = EXPRESSION_FUNCTION;
      expr->d.expression_function_v = fun;
      return expr;
    }
  else if ( ch == 's' || ch == 'S' )
    {
      struct function_s *fun = GC_malloc (sizeof (struct function_s));
      struct expression_s *expr = GC_malloc (sizeof (struct expression_s));

      fun->t = FUNCTION_S;
      expr->t = EXPRESSION_FUNCTION;
      expr->d.expression_function_v = fun;
      return expr;
    }
  else if ( ch == 'v' || ch == 'V' )
    {
      struct function_s *fun = GC_malloc (sizeof (struct function_s));
      struct expression_s *expr = GC_malloc (sizeof (struct expression_s));

      fun->t = FUNCTION_V;
      expr->t = EXPRESSION_FUNCTION;
      expr->d.expression_function_v = fun;
      return expr;
    }
  else if ( ch == 'd' || ch == 'D' )
    {
      struct function_s *fun = GC_malloc (sizeof (struct function_s));
      struct expression_s *expr = GC_malloc (sizeof (struct expression_s));

      fun->t = FUNCTION_D;
      expr->t = EXPRESSION_FUNCTION;
      expr->d.expression_function_v = fun;
      return expr;
    }
  else if ( ch == 'c' || ch == 'C' )
    {
      struct function_s *fun = GC_malloc (sizeof (struct function_s));
      struct expression_s *expr = GC_malloc (sizeof (struct expression_s));

      fun->t = FUNCTION_C;
      expr->t = EXPRESSION_FUNCTION;
      expr->d.expression_function_v = fun;
      return expr;
    }
  else if ( ch == 'e' || ch == 'E' )
    {
      struct function_s *fun = GC_malloc (sizeof (struct function_s));
      struct expression_s *expr = GC_malloc (sizeof (struct expression_s));

      fun->t = FUNCTION_C;
      expr->t = EXPRESSION_FUNCTION;
      expr->d.expression_function_v = fun;
      return expr;
    }
  else if ( ch == 'r' )
    {
      struct function_s *fun = GC_malloc (sizeof (struct function_s));
      struct expression_s *expr = GC_malloc (sizeof (struct expression_s));

      fun->t = FUNCTION_DOT;
      fun->d.function_dot_v = '\n';
      expr->t = EXPRESSION_FUNCTION;
      expr->d.expression_function_v = fun;
      return expr;
    }
  else if ( ch == '.' )
    {
      struct function_s *fun = GC_malloc (sizeof (struct function_s));
      struct expression_s *expr = GC_malloc (sizeof (struct expression_s));
      int ch2;

      fun->t = FUNCTION_DOT;
      ch2 = getc (input);
      if ( ch2 == EOF )
	goto ueof;
      fun->d.function_dot_v = ch2;
      expr->t = EXPRESSION_FUNCTION;
      expr->d.expression_function_v = fun;
      return expr;
    }
  else if ( ch == '@' )
    {
      struct function_s *fun = GC_malloc (sizeof (struct function_s));
      struct expression_s *expr = GC_malloc (sizeof (struct expression_s));

      fun->t = FUNCTION_AT;
      expr->t = EXPRESSION_FUNCTION;
      expr->d.expression_function_v = fun;
      return expr;
    }
  else if ( ch == '?' )
    {
      struct function_s *fun = GC_malloc (sizeof (struct function_s));
      struct expression_s *expr = GC_malloc (sizeof (struct expression_s));
      int ch2;

      fun->t = FUNCTION_QUES;
      ch2 = getc (input);
      if ( ch2 == EOF )
	goto ueof;
      fun->d.function_ques_v = ch2;
      expr->t = EXPRESSION_FUNCTION;
      expr->d.expression_function_v = fun;
      return expr;
    }
  else if ( ch == '|' )
    {
      struct function_s *fun = GC_malloc (sizeof (struct function_s));
      struct expression_s *expr = GC_malloc (sizeof (struct expression_s));

      fun->t = FUNCTION_PIPE;
      expr->t = EXPRESSION_FUNCTION;
      expr->d.expression_function_v = fun;
      return expr;
    }
  else if ( ch == EOF )
    {
    ueof:
      fprintf (stderr, "Unexpected end of file\n");
      exit (1);
    }
  else
    {
      fprintf (stderr, "Character not recognized: %c\n", ch);
      exit (1);
    }
  return NULL;
}

int
main (int argc, char *argv[])
{
  struct expression_s *expr;
  struct continuation_s *finis = GC_malloc (sizeof (struct continuation_s));
  struct task_s *task = GC_malloc (sizeof (struct task_s));

  if ( argc == 1 )
    expr = parse (stdin);
  else if ( argc == 2 )
    {
      FILE *f;
      f = fopen (argv[1], "r");
      if ( ! f )
	{
	  perror ("Can't open input file");
	  exit (1);
	}
      expr = parse (f);
      fclose (f);
    }
  else
    {
      fprintf (stderr, "Expected zero or one argument");
      exit (1);
    }
  finis->t = CONTINUATION_FINAL;
  task->t = TASK_EVAL;
  task->d.task_eval_v.expr = expr;
  task->d.task_eval_v.cont = finis;
  while ( task->t != TASK_FINAL )
    task = run (task);
  return 0;
}
